!
! This subroutine was designed for testing the influence of clumping on Type IIb SN. It allows
! the CL factor to be automatically increased so that a smooth convergence can be obtained.
!
	SUBROUTINE AUTO_CLUMP_REV(POPS,CLUMP_LAW,CLUMP_PAR,N_CLUMP_PAR,DID_REVISION,ND,NT,LUIN)
	USE SET_KIND_MODULE
	USE MOD_CMFGEN
	USE UPDATE_KEYWORD_INTERFACE
	IMPLICIT NONE
!
! Altered 11-Mar-2014: CALL TO REV_SN_HYDRO updated. Updated temperature now output to SN_HYDRO.
! Altered 15-Feb-2023: REXP -> PEXP to allow non-unity factor at infinity (PC)
! requiring an extra (4th) CLUMP_PAR with respect to REXP
!
	INTEGER ND
	INTEGER NT
	INTEGER LUIN
!
	REAL(KIND=LDP) POPS(NT,ND)
	LOGICAL DID_REVISION
!
	INTEGER N_CLUMP_PAR
	REAL(KIND=LDP) CLUMP_PAR(N_CLUMP_PAR)
	CHARACTER(LEN=*) CLUMP_LAW
!
	REAL(KIND=LDP) OLD_CLUMP_FAC(ND)
	REAL(KIND=LDP) CURRENT_CLUMP_VALUE
	REAL(KIND=LDP) NEW_CLUMP_VALUE
	REAL(KIND=LDP) DESIRED_CLUMP_VALUE
	REAL(KIND=LDP) CLUMP_REV_FACTOR
	REAL(KIND=LDP) SCALE_FACTOR
!
	INTEGER ISPEC
	INTEGER IOS
	INTEGER ID
	INTEGER I,L
	LOGICAL, PARAMETER :: L_TRUE=.TRUE.
!
	OPEN(UNIT=LUIN,FILE='CLUMP_VALUES',IOSTAT=IOS,STATUS='OLD')
	  IF(IOS .NE. 0)THEN
	    WRITE(6,*)'Unable to open fle with clump values'
	    DID_REVISION=.FALSE.
	    RETURN
	  END IF
	  READ(LUIN,*,IOSTAT=IOS,ERR=100)CURRENT_CLUMP_VALUE
	  READ(LUIN,*,IOSTAT=IOS,ERR=100)DESIRED_CLUMP_VALUE
	  READ(LUIN,*,IOSTAT=IOS,ERR=100)CLUMP_REV_FACTOR
100	  IF(IOS .NE. 0)THEN
	    WRITE(6,*)'Error reading CLUMP_VALUES: IOSTAT=',IOS
	    CLOSE(LUIN)
	    RETURN
	  END IF
	  IF(DESIRED_CLUMP_VALUE .EQ. CURRENT_CLUMP_VALUE)THEN
	    CLOSE(LUIN)
	    DID_REVISION=.FALSE.
	    RETURN
	  END IF
	CLOSE(LUIN)
	NEW_CLUMP_VALUE=MAX(CURRENT_CLUMP_VALUE/CLUMP_REV_FACTOR,DESIRED_CLUMP_VALUE)
	CLUMP_PAR(1)=NEW_CLUMP_VALUE
!
	CALL REV_CURRENT(CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
	CALL REV_SN_HYDRO(V,T,ND,CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
	CALL REV_OLD_MODEL(CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
!
	CALL SYSTEM('mv -f new_current current_nonlocal_decay_energy.dat')
	CALL SYSTEM('mv -f NEW_MODEL_DATA OLD_MODEL_DATA')
	CALL SYSTEM('mv -f NEW_SN SN_HYDRO_DATA')
!
! As we have changed all the files, we update keywords.
!
	OPEN(UNIT=LUIN,FILE='CLUMP_VALUES',IOSTAT=IOS)
	  WRITE(LUIN,*)NEW_CLUMP_VALUE
	  WRITE(LUIN,*)DESIRED_CLUMP_VALUE
	  WRITE(LUIN,*)CLUMP_REV_FACTOR
	CLOSE(LUIN)
!
	DID_REVISION=.TRUE.
	CALL UPDATE_KEYWORD(CLUMP_PAR(1),'[CL_PAR_1]','VADAT',L_TRUE,L_TRUE,LUIN)
!
	OLD_CLUMP_FAC(1:ND)=CLUMP_FAC(1:ND)
	CALL DO_CLUMP_REV(V,ND,CLUMP_FAC, CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
!
	DO L=1,ND
	  SCALE_FACTOR=OLD_CLUMP_FAC(L)/CLUMP_FAC(L)
	  DO I=1,NT-1
	    POPS(I,L)=POPS(I,L)*SCALE_FACTOR
	  END DO
	  ED(L)=ED(L)*SCALE_FACTOR
	END DO
!
	DO ISPEC=1,NUM_SPECIES
	  DO ID=SPECIES_BEG_ID(ISPEC),SPECIES_END_ID(ISPEC)-1
	    DO L=1,ND
	      SCALE_FACTOR=OLD_CLUMP_FAC(L)/CLUMP_FAC(L)
	      ATM(ID)%XzV(:,L)=ATM(ID)%XzV(:,L)*SCALE_FACTOR
	      ATM(ID)%XzV_F(:,L)=ATM(ID)%XzV_F(:,L)*SCALE_FACTOR
	      ATM(ID)%DXzV_F(L)=ATM(ID)%DXzV_F(L)*SCALE_FACTOR
	      ATM(ID)%DXzV(L)=ATM(ID)%DXzV(L)*SCALE_FACTOR
	    END DO
	  END DO
	END DO
!
	RETURN
	END
!
	SUBROUTINE REV_CURRENT(CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
	USE SET_KIND_MODULE
	IMPLICIT NONE
!
! Altered: 15-Nov-2021 : V changed to vector for consistency in call.
!                            Now use GET_LU for LUIN and LUOUT
!                            Now close units LUIN and LUOUT
!
	INTEGER N_CLUMP_PAR
	REAL(KIND=LDP) CLUMP_PAR(N_CLUMP_PAR)
	CHARACTER(LEN=*) CLUMP_LAW
!
	INTEGER, PARAMETER :: IONE=1
	INTEGER I,ND,LUIN,LUOUT
	REAL(KIND=LDP) V_VEC(1),E1,E2
	REAL(KIND=LDP) NEW_CLUMP_VALUE(1)
	CHARACTER(LEN=80) STRING
!
	CALL GET_LU(LUIN, 'In REV_CURRENT - LUIN')
	CALL GET_LU(LUOUT,'In REV_CURRENT - LUOUT')
	OPEN(UNIT=LUIN, FILE='orig_current_nonlocal_decay_energy.dat',STATUS='OLD',ACTION='READ')
	OPEN(UNIT=LUOUT,FILE='new_current',STATUS='UNKNOWN',ACTION='WRITE')
!
	READ(LUIN,'(A)')STRING
	READ(STRING,*)ND
	WRITE(LUOUT,'(A)')TRIM(STRING)
	READ(LUIN,'(A)')STRING
	WRITE(LUOUT,'(A)')TRIM(STRING)
!
	DO I=1,ND
	  READ(LUIN,*)V_VEC(1),E1,E2
	  CALL DO_CLUMP_REV(V_VEC,IONE,NEW_CLUMP_VALUE, CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
	  WRITE(LUOUT,*)V_VEC(1),E1/NEW_CLUMP_VALUE(1),E2/NEW_CLUMP_VALUE(1)
	END DO
	CLOSE(LUIN)
	CLOSE(LUOUT)
!
	RETURN
	END
!
	SUBROUTINE REV_OLD_MODEL(CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
	USE SET_KIND_MODULE
	IMPLICIT NONE
!
	INTEGER N_CLUMP_PAR
	REAL(KIND=LDP) CLUMP_PAR(N_CLUMP_PAR)
	CHARACTER(LEN=*) CLUMP_LAW
!
	REAL(KIND=LDP), ALLOCATABLE :: R(:)
	REAL(KIND=LDP), ALLOCATABLE :: V(:)
	REAL(KIND=LDP), ALLOCATABLE :: SIGMA(:)
	REAL(KIND=LDP), ALLOCATABLE :: CLUMP_FAC(:)
	REAL(KIND=LDP), ALLOCATABLE :: T(:)
	REAL(KIND=LDP), ALLOCATABLE :: ED(:)
	REAL(KIND=LDP), ALLOCATABLE :: POP_ATOM(:)
	REAL(KIND=LDP), ALLOCATABLE :: DENSITY(:)
	REAL(KIND=LDP), ALLOCATABLE :: DXzV(:)
	REAL(KIND=LDP), ALLOCATABLE :: XZV_F(:,:)
!
	INTEGER SPECIES_BEG_ID
	INTEGER SPECIES_END_ID
!
	INTEGER, PARAMETER :: LU_RD=10
	INTEGER, PARAMETER :: LU_WR=12
!
	INTEGER NUM_SPECIES
	INTEGER ND
	INTEGER NXzV_F
!
	REAL(KIND=LDP) NEW_CLUMP_VALUE
	REAL(KIND=LDP) SN_AGE
	REAL(KIND=LDP) ZXzV
!
	INTEGER L
	INTEGER ISPEC
	INTEGER ID
	INTEGER ID_RD
!
	CHARACTER(LEN=10) SPECIES
	CHARACTER(LEN=11) DATE
!
	OPEN(UNIT=LU_RD,FILE='ORIG_OLD_MODEL_DATA',FORM='UNFORMATTED',STATUS='OLD',ACTION='READ')
	OPEN(UNIT=LU_WR,FILE='NEW_MODEL_DATA',FORM='UNFORMATTED',STATUS='UNKNOWN',ACTION='WRITE')
!
	  READ(LU_RD)DATE
	  WRITE(LU_WR)DATE
!
	  READ(LU_RD)ND,NUM_SPECIES
	  WRITE(LU_WR)ND,NUM_SPECIES
!
	  READ(LU_RD)SN_AGE
	  WRITE(LU_WR)SN_AGE
!
	  ALLOCATE (R(ND),V(ND),SIGMA(ND),CLUMP_FAC(ND))	
	  ALLOCATE (T(ND),ED(ND),POP_ATOM(ND),DENSITY(ND))	
!
	  READ(LU_RD)R
	  READ(LU_RD)V
	  READ(LU_RD)SIGMA
	  READ(LU_RD)T
	  READ(LU_RD)ED
	  READ(LU_RD)POP_ATOM
	  READ(LU_RD)DENSITY
	  CALL DO_CLUMP_REV(V, ND, CLUMP_FAC, CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
!
	  WRITE(LU_WR)R
	  WRITE(LU_WR)V
	  WRITE(LU_WR)SIGMA
	  WRITE(LU_WR)T
	  WRITE(LU_WR)ED/CLUMP_FAC
	  WRITE(LU_WR)POP_ATOM/CLUMP_FAC
	  WRITE(LU_WR)DENSITY/CLUMP_FAC
!
! We use ZXzV as a means of identifying the ionization stage.
!
	  ALLOCATE (DXzV(ND))
	  DO ISPEC=1,NUM_SPECIES
	    READ(LU_RD)SPECIES_BEG_ID,SPECIES_END_ID,SPECIES
	    WRITE(LU_WR)SPECIES_BEG_ID,SPECIES_END_ID,SPECIES
	    DO ID=SPECIES_BEG_ID,SPECIES_END_ID
	      READ(LU_RD)ID_RD,NXzV_F,ZXzV
	      WRITE(LU_WR)ID_RD,NXzV_F,ZXzV
	      ALLOCATE(XzV_F(NXzV_F,ND))
	      READ(LU_RD)XzV_F
	      DO L=1,ND
	        XzV_F(:,L)=XzV_F(:,L)/CLUMP_FAC(L)
	      END DO
	      WRITE(LU_WR)XzV_F
	      READ(LU_RD)DXzV
	      DO L=1,ND
	        DXzV(L)=DXzV(L)/CLUMP_FAC(L)
	      END DO
	      WRITE(LU_WR)DXzV			!Super level ion population.
	      DEALLOCATE(XzV_F)
	    END DO
	  END DO
!
	  DEALLOCATE (R,V,SIGMA,CLUMP_FAC,T,ED,POP_ATOM,DENSITY,DXzV)	
!
	CLOSE(LU_RD)
	CLOSE(LU_WR)
	RETURN
	END
!
	SUBROUTINE REV_SN_HYDRO(V,T,ND,CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
	USE SET_KIND_MODULE
	IMPLICIT NONE
!
	INTEGER ND
	INTEGER N_CLUMP_PAR
	REAL(KIND=LDP) V(ND)
	REAL(KIND=LDP) T(ND)
	REAL(KIND=LDP) CLUMP_PAR(N_CLUMP_PAR)
	CHARACTER(LEN=*) CLUMP_LAW
!
	REAL(KIND=LDP), ALLOCATABLE :: TA(:)
	REAL(KIND=LDP), ALLOCATABLE :: V_SN(:)
	REAL(KIND=LDP), ALLOCATABLE :: CLUMP_FAC(:)
	REAL(KIND=LDP), ALLOCATABLE :: T_SN(:)
!
	INTEGER ND_SN
	INTEGER I
	INTEGER, PARAMETER :: IONE=1
	CHARACTER(LEN=132) STRING
!
	OPEN(UNIT=10,FILE='ORIG_SN_HYDRO_DATA',STATUS='OLD',ACTION='READ')
	OPEN(UNIT=12,FILE='NEW_SN',STATUS='UNKNOWN',ACTION='WRITE')
!
	READ(10,'(A)')STRING
	WRITE(12,'(A)')TRIM(STRING)
	READ(10,'(A)')STRING
	WRITE(12,'(A)')TRIM(STRING)
	I=INDEX(STRING,':')
	READ(STRING(I+1:),*)ND_SN
	ALLOCATE (TA(ND_SN),V_SN(ND_SN),CLUMP_FAC(ND_SN))
!
! V_SN is needed to compute the clumping factor and to interpolate T.
!
	DO WHILE(INDEX(STRING,'Velocity') .EQ. 0)
	  READ(10,'(A)')STRING
	  WRITE(12,'(A)')TRIM(STRING)
	END DO
	READ(10,*)V_SN
	WRITE(12,'(X,8ES16.7)')V_SN
	CALL DO_CLUMP_REV(V_SN, ND_SN, CLUMP_FAC, CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
!
	DO WHILE(INDEX(STRING,'Temperature') .EQ. 0)
	  READ(10,'(A)')STRING
	  WRITE(12,'(A)')TRIM(STRING)
	END DO
	ALLOCATE (T_SN(ND_SN))
	READ(10,*)T_SN			!Values to be overwritten
	CALL MON_INTERP(T_SN,ND_SN,IONE,V_SN,ND_SN,T,ND,V,ND)
	WRITE(12,'(X,8ES16.7)')T_SN
!
	DO WHILE(INDEX(STRING,'Density') .EQ. 0)
	  READ(10,'(A)')STRING
	  WRITE(12,'(A)')TRIM(STRING)
	END DO
	READ(10,*)TA
	TA=TA/CLUMP_FAC
	WRITE(12,'(X,8ES16.7)')TA
!
	DO WHILE(INDEX(STRING,'Atom density') .EQ. 0)
	  READ(10,'(A)')STRING
	  WRITE(12,'(A)')TRIM(STRING)
	END DO
	READ(10,*)TA
	TA=TA/CLUMP_FAC
	WRITE(12,'(X,8ES16.7)')TA
!
	DO WHILE(INDEX(STRING,'Electron density') .EQ. 0)
	  READ(10,'(A)')STRING
	  WRITE(12,'(A)')TRIM(STRING)
	END DO
	READ(10,*)TA
	TA=TA/CLUMP_FAC
	WRITE(12,'(X,8ES16.7)')TA
!
	DO WHILE(INDEX(STRING,'Rosseland mean') .EQ. 0)
	  READ(10,'(A)')STRING
	  WRITE(12,'(A)')TRIM(STRING)
	END DO
	READ(10,*)TA
	TA=TA/CLUMP_FAC
	WRITE(12,'(X,8ES16.7)')TA
!
	DO WHILE(1 .EQ. 1)
	  READ(10,'(A)',END=100)STRING
	  WRITE(12,'(A)')TRIM(STRING)
	END DO
100	CONTINUE
!
	DEALLOCATE (TA,V_SN,CLUMP_FAC,T_SN)
!
	RETURN
	END
!
	SUBROUTINE DO_CLUMP_REV(V,ND,CLUMP_FAC, CLUMP_LAW, CLUMP_PAR, N_CLUMP_PAR)
	USE SET_KIND_MODULE
	IMPLICIT NONE
!
	INTEGER ND
	REAL(KIND=LDP) V(ND)
!
	REAL(KIND=LDP) CLUMP_FAC(ND)
	INTEGER N_CLUMP_PAR
	REAL(KIND=LDP) CLUMP_PAR(N_CLUMP_PAR)
	CHARACTER(LEN=*) CLUMP_LAW
!
	INTEGER K
!
	IF(CLUMP_LAW(1:4) .EQ. 'EXPO')THEN
	  IF(CLUMP_PAR(3) .EQ. 0.0_LDP)CLUMP_PAR(4)=1.0_LDP
	  DO K=1,ND
	    CLUMP_FAC(K)=CLUMP_PAR(1)+(1.0_LDP-CLUMP_PAR(1)-CLUMP_PAR(3))*
	1                     EXP(-V(K)/CLUMP_PAR(2))+
	1                     CLUMP_PAR(3)*EXP(-V(K)/CLUMP_PAR(4))
	  END DO
	ELSE IF(CLUMP_LAW(1:4) .EQ. 'REXP')THEN
	  DO K=1,ND
	    CLUMP_FAC(K)=CLUMP_PAR(1)+(1.0_LDP-CLUMP_PAR(1))*
	1           EXP(-V(K)/CLUMP_PAR(2))+
	1           (1.0_LDP-CLUMP_PAR(1))*EXP( (V(K)-V(1))/CLUMP_PAR(3))
	  END DO
	ELSE IF(CLUMP_LAW(1:4) .EQ. 'PEXP')THEN
	  DO K=1,ND
	    CLUMP_FAC(K)=CLUMP_PAR(1)+(1.0_LDP-CLUMP_PAR(1))*
	1           EXP(-V(K)/CLUMP_PAR(2))+
	1           (CLUMP_PAR(4)-CLUMP_PAR(1))*EXP( (V(K)-V(1))/CLUMP_PAR(3))
	  END DO

	ELSE IF(CLUMP_LAW(1:6) .EQ. 'SPLINE')THEN
	  CALL SPL_CLUMP(CLUMP_FAC,V,ND)
!
	ELSE
	  WRITE(6,*)'Error in AUTO_CLUMP_REV'
	  WRITE(6,*)'Invalid law for computing clumping factor'
	  STOP
	END IF
!
	RETURN
	END
